<?php

namespace App\Http\Controllers;

use App\Imports\EmoImport;
use App\Models\EmoRecord;
use App\Models\EmoUpload;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class EmoUploadController extends Controller
{
    
	
	 public function dashboard()
    {
		
			 $today = Carbon::today();

    // Today
    $uploadsToday = EmoUpload::whereDate('created_at', $today)->count();
    $recordsToday = EmoRecord::whereDate('created_at', $today)->count();

    // Past (before today)
    $uploadsPast = EmoUpload::whereDate('created_at', '<', $today)->count();
    $recordsPast = EmoRecord::whereDate('created_at', '<', $today)->count();

    return view('emo.dashboard', compact(
        'uploadsToday', 'recordsToday',
        'uploadsPast', 'recordsPast'
    ));
	

    }


    // List past uploads
    public function index()
    {
        $uploads = EmoUpload::withCount('records')
            ->where('user_id', auth()->id())
            ->latest()->get();

        return view('emo.index', compact('uploads'));
    }

    // Upload form
    public function create()
    {
        return view('emo.upload');
    }

    // Handle Excel upload + import
    public function store(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv|max:10240',
        ]);

        $path = $request->file('file')->store('uploads');

        $upload = EmoUpload::create([
            'user_id'       => auth()->id(),
            'original_name' => $request->file('file')->getClientOriginalName(),
            'file_name'   => $path,
        ]);

        Excel::import(new EmoImport($upload->id), $path);

        return redirect()->route('emo.show', $upload)->with('success', 'Excel imported successfully.');
    }

    // Show one upload with actions
    public function show(EmoUpload $upload)
    {
        $this->authorizeOwner($upload);
        $records = $upload->records()->orderBy('id')->get();
        return view('emo.show', compact('upload', 'records'));
    }

    // Mass print (browser print) - A3
    public function print(EmoUpload $upload)
    {
        $this->authorizeOwner($upload);
        $records = $upload->records()->orderBy('id')->get();
        return view('emo.print', compact('upload', 'records'));
    }

    // Export all receipts in one PDF (A3)
    public function pdf(EmoUpload $upload)
    {
		set_time_limit(300); // 300 seconds (5 mins)

        $this->authorizeOwner($upload);
        $records = $upload->records()->orderBy('id')->get();

        $pdf = Pdf::loadView('emo.pdf', compact('upload', 'records'))
            ->setPaper('a3', 'portrait');

        return $pdf->download('receipts_upload_'.$upload->id.'.pdf');
    }

    protected function authorizeOwner(EmoUpload $upload): void
    {
        abort_if($upload->user_id !== auth()->id(), 403);
    }
}
