<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Mail\Ordercreated;
use App\Models\BranchModel;
use App\Models\CategoryModel;
use App\Models\ClientAddressModel;
use App\Models\ClientModel;
use App\Models\OrdersItemsModel;
use App\Models\OrdersModel;
use App\Models\ProductDetailsModel;
use App\Models\ProductModel;
use App\Models\ProductPaymentModel;
use App\Models\PurchaseModel;
use App\Models\RightsModel;
use App\Models\SupplierModel;
use App\Models\TempQuantityModel;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Mpdf\Mpdf;

class MobOrderController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();

        $query = OrdersModel::with([
            'client' => function ($query) {
                $query->select('id', 'user_id');
            },
            'client.user' => function ($query) {
                $query->select('id', 'name', 'phone');
            },
            'branch',
            'ordersItem'
        ])->orderBy('id', 'desc');

        if ($request->filled('branch_id')) {
            $query->where('branch_id', $request->branch_id);
        }

        if ($user->user_flg === 'C') {
            $client_id = ClientModel::where('user_id', $user->id) // Corrected: use $user->id, not $user->user_id
                ->when($request->filled('branch_id'), function ($q) use ($request) {
                    $q->where('branch_id', $request->branch_id);
                })
                ->pluck('id')
                ->first();

            if ($client_id) {
                $query->where('client_id', $client_id);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Invalid client/user',
                ], 400);
            }
        }

        if ($request->filled('search')) {
            $searchTerm = $request->search;

            $query->where(function ($q) use ($searchTerm) {
                $q->where('order_id', 'like', "%{$searchTerm}%")
                    ->orWhereHas('client.user', function ($userQuery) use ($searchTerm) {
                        $userQuery->where('name', 'like', "%{$searchTerm}%")
                            ->orWhere('phone', 'like', "%{$searchTerm}%");
                    });
            });
        }

        $perPage = $request->get('per_page', 10);

        return response()->json($query->paginate($perPage));
    }


    public function getClientAddress($phone)
    {
        $user_id = User::where('phone', $phone)->value('id');
        $client = ClientModel::with('clientAddress')->where('user_id', $user_id)->value('id');
        $clientID = isset($client) ? $client : $phone;
        $client_address = ClientAddressModel::where('client_id', $clientID)->get();

        return response()->json($client_address);
    }

    public function getClients(Request $request)
    {
        if (Auth::user()->user_flg == 'C') {
            $clients = ClientModel::with(['user:id,name,phone'])
                ->where('user_id', Auth::user()->id)
                ->get();
        } else {
            $clients = ClientModel::with(['user:id,name,phone'])
                ->orderBy('id', 'desc')
                ->get();
        }
        return response()->json($clients);
    }

    public function store(Request $request)
    {
        $authUser = Auth::user();
        $request->validate([
            'products.*.qty' => 'required|numeric',
            'products.*.unit_price' => 'required|numeric',
            'products.*.amount' => 'required|numeric',
            'client_num' => 'required',
        ]);

        DB::beginTransaction();

        try {
            $user = User::where('phone', $request->client_num)->first();

            $client_id = $this->saveClient($request, $user);

            if (!$client_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create client'
                ], 400);
            }

            $address_id = null;
            if ($request->has('address_id')) {
                $address_id = $request->address_id;
            } elseif ($request->has('address')) {
                $address = ClientAddressModel::create([
                    'client_id' => $client_id,
                    'address' => $request->address,
                    'is_default' => $request->save_address ? 'Y' : 'N',
                    'oprtnl_flag' => 'A',
                    'created_by' => $authUser->id
                ]);
                $address_id = $address->id;
            }

            $lastOrder = OrdersModel::orderBy('id', 'desc')->first();
            $order_id = $this->generateOrderId($lastOrder);

            $order = OrdersModel::create([
                'order_id' => $order_id,
                'client_id' => $client_id,
                'order_date' => $request->order_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'note' => $request->note,
                'term' => $request->term,
                'branch_id' => $request->branch_id,
                'status' => $request->status ?? '',
                'address_id' => $address_id
            ]);

            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                OrdersItemsModel::create([
                    'order_id' => $order->id,
                    'product_id' => $field['id'],
                    'product_name' => $product->name ?? '',
                    'available_quantity' => $product->qty,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                ]);
            }
            $settings = BranchModel::where('id', $request->branch_id)->first();

            $orderDetails = OrdersModel::with('client.clientAddress', 'ordersItem.product')
                ->where('id', $order->id)->where('branch_id', $request->branch_id)->first();
            $data = [
                'addedOrder' => $orderDetails,
                'settings' => $settings
            ];

            $html = view('orders.email-pdf.order-generate-pdf_email', $data)->render();
            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);
            $pdfContent = $mpdf->Output('', 'S');
            $admin_mail = User::where('user_flg', 'A')->where('branch_id', $request->branch_id)->pluck('email')->toArray();
            Mail::to($admin_mail)->send(new Ordercreated($pdfContent, $order, $settings));

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Order created successfully',
                'data' => [
                    'order_id' => $order->id,
                    'order_number' => $order->order_id
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error creating order: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to create order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $authUser = Auth::user();
        $request->validate([
            'products.*.qty' => 'required|numeric',
            'products.*.unit_price' => 'required|numeric',
            'products.*.amount' => 'required|numeric',
            'client_num' => 'required',
        ]);

        DB::beginTransaction();

        try {
            $order = OrdersModel::findOrFail($id);
            $user = User::where('phone', $request->client_num)->first();

            $client_id = $this->saveClient($request, $user);

            if (!$client_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update client information'
                ], 400);
            }

            $address_id = $order->address_id;
            if ($request->has('address_id')) {
                $address_id = $request->address_id;
            } elseif ($request->has('address')) {
                // If new address is provided
                $address = ClientAddressModel::create([
                    'client_id' => $client_id,
                    'address' => $request->address,
                    'is_default' => $request->save_address ? 'Y' : 'N',
                    'oprtnl_flag' => 'A',
                    'created_by' => $authUser->id
                ]);
                $address_id = $address->id;
            }

            $order->update([
                'client_id' => $client_id,
                'order_date' => $request->order_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'note' => $request->note,
                'term' => $request->term,
                'branch_id' => $request->branch_id,
                'status' => $request->status ?? '',
                'address_id' => $address_id
            ]);

            // Delete existing order items
            OrdersItemsModel::where('order_id', $order->id)->delete();

            // Add new order items
            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                OrdersItemsModel::create([
                    'order_id' => $order->id,
                    'product_id' => $field['id'],
                    'product_name' => $product->name ?? '',
                    'available_quantity' => $product->qty,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                ]);
            }

            // Get updated order details
            $orderDetails = OrdersModel::with('client.clientAddress', 'ordersItem.product')
                ->where('id', $order->id)
                ->where('branch_id', $request->branch_id)
                ->first();

            $settings = BranchModel::where('id', $request->branch_id)->first();
            $data = [
                'addedOrder' => $orderDetails,
                'settings' => $settings
            ];

            // Generate updated PDF
            $html = view('orders.email-pdf.order-generate-pdf_email', $data)->render();
            $mpdf = new Mpdf([
                'mode' => 'utf-8',
                'format' => 'A4',
                'default_font' => 'freeserif',
                'autoScriptToLang' => true,
                'autoLangToFont' => true,
            ]);

            $mpdf->WriteHTML($html);
            $pdfContent = $mpdf->Output('', 'S');
            $admin_mail = User::where('user_flg', 'A')->where('branch_id', $request->branch_id)->pluck('email')->toArray();
            Mail::to($admin_mail)->send(new Ordercreated($pdfContent, $order, $settings));

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Order updated successfully',
                'data' => [
                    'order_id' => $order->id,
                    'order_number' => $order->order_id
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error updating order: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to update order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    protected function generateOrderId($lastorder)
    {
        if ($lastorder) {
            $lastOrderId = $lastorder->order_id;
            $parts = explode('-', $lastOrderId);
            $prefix = $parts[0];
            $number = (int) $parts[1];
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }
        return 'OSORD-001';
    }

    protected function saveClient(Request $request, $user = null)
    {

        try {
            if ($user) {
                $client = ClientModel::where('user_id', $user->id)->first();
                return $client->id;
            }

            $randomEmail = Str::random(10) . '@billing.com';

            $right = RightsModel::where('rights_type', 'client')->first();
            $clientRight = $right ? $right->id : null;
            $user = User::create([
                'name' => $request->client_name,
                'email' => $randomEmail,
                'phone' => $request->client_num,
                'password' => Hash::make('client@123'),
                'user_flg' => 'C',
                'active_status' => 'A',
                'branch_id' => $request->branch_id,
                'rights_id' => $clientRight,

            ]);

            $client = ClientModel::create([
                'user_id' => $user->id,
                'branch_id' => $request->branch_id,
                'active_status' => 'A',
            ]);
            $allBranchIds = DB::table('branch')->pluck('id');

            $insertData = [];

            foreach ($allBranchIds as $branchId) {
                $insertData[] = [
                    'user_id' => $user->id,
                    'branch_id' => $branchId,
                ];
            }

            DB::table('branch_user')->insert($insertData);
            return $client->id;
        } catch (\Exception $e) {
            \Log::info('While create Client:' . $e);
            return null;
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            $order = OrdersModel::findOrFail($id);

            // Delete all associated order items first
            OrdersItemsModel::where('order_id', $order->id)->delete();

            // Then delete the order
            $order->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Order deleted successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error deleting order: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete order',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // Updated method in MobOrderController.php
    public function getDailyRequirements(Request $request)
    {
        Log::info('getDailyRequirements called with params: ' . json_encode($request->all()));
        try {
            $date = $request->get('date', now()->toDateString());
            $branchId = $request->branch_id;

            // Get aggregated data for all unconverted orders
            $requirements = DB::table('order_items')
                ->join('orders', 'order_items.order_id', '=', 'orders.id')
                ->join('products', 'order_items.product_id', '=', 'products.id')
                ->leftJoin('temp_quantities', function ($join) use ($branchId) {
                    $join->on('order_items.product_id', '=', 'temp_quantities.product_id')
                        ->where('temp_quantities.branch_id', '=', $branchId);
                })
                ->select([
                    'order_items.product_id',
                    'order_items.product_name',
                    'products.qty as current_available_qty',
                    DB::raw('SUM(order_items.quantity) as total_ordered_qty'),
                    DB::raw('CASE 
                    WHEN SUM(order_items.quantity) > products.qty 
                    THEN SUM(order_items.quantity) - products.qty 
                    ELSE 0 
                END as shortage_qty'),
                    DB::raw('COALESCE(temp_quantities.temp_qty, 0) as temp_qty'),
                    DB::raw('COUNT(DISTINCT orders.id) as total_orders'),
                    DB::raw('MIN(orders.order_date) as earliest_order_date'),
                    DB::raw('MAX(orders.order_date) as latest_order_date')
                ])
                ->where('orders.branch_id', $branchId)
                ->where('orders.active_flag', 'A')
                ->where('order_items.active_flag', 'A')
                ->where(function ($query) {
                    $query->where('orders.oprntl_flag', '!=', 'C')
                        ->orWhereNull('orders.oprntl_flag');
                })
                ->groupBy([
                    'order_items.product_id',
                    'order_items.product_name',
                    'products.qty',
                    'temp_quantities.temp_qty'
                ])
                ->orderByDesc('shortage_qty')
                ->orderBy('order_items.product_name')
                ->get();

            // Process results
            $processedRequirements = [];
            foreach ($requirements as $item) {
                $processedRequirements[] = [
                    'product_id' => (int) $item->product_id,
                    'product_name' => $item->product_name,
                    'available_quantity' => (int) $item->current_available_qty,
                    'total_ordered_qty' => (int) $item->total_ordered_qty,
                    'current_available_qty' => (int) $item->current_available_qty,
                    'shortage_qty' => (int) $item->shortage_qty,
                    'temp_qty' => (int) $item->temp_qty,
                    'total_orders' => (int) $item->total_orders,
                    'earliest_order_date' => $item->earliest_order_date,
                    'latest_order_date' => $item->latest_order_date,
                    'date_range' => $item->earliest_order_date === $item->latest_order_date
                        ? Carbon::parse($item->earliest_order_date)->format('M d, Y')
                        : Carbon::parse($item->earliest_order_date)->format('M d') . ' - ' . Carbon::parse($item->latest_order_date)->format('M d, Y')
                ];
            }

            Log::info('Daily Requirements processed: ' . count($processedRequirements) . ' products from unconverted orders');

            return response()->json([
                'success' => true,
                'date' => $date,
                'data_type' => 'all_unconverted_orders',
                'requirements' => $processedRequirements,
                'total_products' => count($processedRequirements),
                'products_with_shortage' => collect($processedRequirements)->where('shortage_qty', '>', 0)->count(),
                'total_unconverted_orders' => collect($processedRequirements)->sum('total_orders'),
                'total_temp_qty' => collect($processedRequirements)->sum('temp_qty')
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching daily requirements: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch requirements',
                'error' => $e->getMessage()
            ], 500);
        }
    }



    public function updateTempQuantity(Request $request)
    {
        $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'temp_qty' => 'required|integer|min:0',
            'branch_id' => 'required|integer'
        ]);

        try {
            DB::beginTransaction();

            // Update or create temp quantity
            TempQuantityModel::updateOrCreate(
                [
                    'product_id' => $request->product_id,
                    'branch_id' => $request->branch_id
                ],
                [
                    'temp_qty' => $request->temp_qty
                ]
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Temporary quantity updated successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error updating temp quantity: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update temporary quantity',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getTempQuantitiesForPurchase(Request $request)
    {
        try {
            $branchId = $request->branch_id;

            $tempProducts = DB::table('temp_quantities')
                ->join('products', 'temp_quantities.product_id', '=', 'products.id')
                ->leftJoin('categories', 'products.category_id', '=', 'categories.id')
                ->leftJoin('supplier', 'products.supplier_id', '=', 'supplier.id')
                ->where('temp_quantities.branch_id', $branchId)
                ->where('temp_quantities.temp_qty', '>', 0)
                ->where('products.active_flag', 'A')
                ->select([
                    'temp_quantities.id as temp_id',
                    'products.id as product_id',
                    'products.name as product_name',
                    'products.native_name',
                    'products.code',
                    'products.hsn_code',
                    'products.unit_price',
                    'products.margin_price',
                    'products.description',
                    'products.products_image',
                    'temp_quantities.temp_qty',
                    'categories.id as category_id',
                    'categories.name as category_name',
                    'supplier.id as supplier_id',
                    'supplier.supplier_name as supplier_name',
                    'supplier.phone as supplier_mobile'
                ])
                ->orderBy('products.name')
                ->get();

            $suppliers = SupplierModel::where('branch_id', $branchId)
                ->where('active_status', 1)
                ->select('id', 'supplier_name', 'phone')
                ->get();

            $categories = CategoryModel::where('active_flag', 'A')
                ->select('id', 'name')
                ->get();

            return response()->json([
                'success' => true,
                'temp_products' => $tempProducts,
                'suppliers' => $suppliers,
                'categories' => $categories,
                'total_temp_products' => $tempProducts->count()
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching temp quantities: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch temp quantities',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function convertTempToPurchase(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'payment_date' => 'nullable|date',
            'payment_type' => 'nullable|string',
            'total_paid_amount' => 'nullable|numeric|min:0',
            'products' => 'required|array|min:1',
            'products.*.temp_id' => 'required|integer|exists:temp_quantities,id',
            'products.*.product_id' => 'required|integer|exists:products,id',
            'products.*.supplier_id' => 'required|integer|exists:supplier,id',
            'products.*.qty' => 'required|integer|min:1',
            'products.*.unit_price' => 'required|numeric|min:0',
            'products.*.margin_price' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();

            $branchId = $request->branch_id;
            $processedProducts = [];

            // Calculate total amount
            $totalAmount = 0;
            foreach ($request->products as $productData) {
                $totalAmount += $productData['margin_price'] * $productData['qty'];
            }

            // 🔥 CREATE PURCHASE RECORD FIRST
            $purchaseCode = $this->generatePurchaseCode();
            $paidAmount = $request->total_paid_amount ?? 0;
            $dueAmount = $totalAmount - $paidAmount;

            // Determine purchase status
            $purchaseStatus = 'pending';
            if ($paidAmount >= $totalAmount) {
                $purchaseStatus = 'paid';
            } elseif ($paidAmount > 0) {
                $purchaseStatus = 'partial';
            }

            $purchase = PurchaseModel::create([
                'purchase_code' => $purchaseCode,
                'branch_id' => $branchId,
                'purchase_date' => $request->payment_date ?? now()->toDateString(),
                'total_amount' => $totalAmount,
                'paid_amount' => $paidAmount,
                'due_amount' => $dueAmount,
                'status' => $purchaseStatus,
                'notes' => $request->notes ?? 'Converted from temp quantities',
                'active_flag' => 'A'
            ]);

            Log::info("Created purchase: {$purchase->purchase_code} with ID: {$purchase->id}");

            // Process each product and link to purchase
            foreach ($request->products as $productData) {
                // Get temp quantity record
                $tempQuantity = TempQuantityModel::find($productData['temp_id']);
                if (!$tempQuantity || $productData['qty'] > $tempQuantity->temp_qty) {
                    throw new \Exception("Invalid quantity for product ID: {$productData['product_id']}");
                }

                // Update main product stock and prices
                $product = ProductModel::findOrFail($productData['product_id']);
                $product->update([
                    'qty' => $product->qty + $productData['qty'],
                    'total_qty' => ($product->total_qty ?? 0) + $productData['qty'],
                    'unit_price' => $productData['unit_price'],      // selling price
                    'margin_price' => $productData['margin_price'],  // buying price
                    'supplier_id' => $productData['supplier_id'],
                ]);

                // Create product detail entry LINKED TO PURCHASE
                $details_code = $this->generateProductDetailCode();
                $itemTotalAmount = $productData['margin_price'] * $productData['qty'];

                // Determine product detail status based on purchase status
                $itemStatus = 1; // Unpaid
                if ($purchaseStatus === 'paid') {
                    $itemStatus = 2; // Fully Paid
                } elseif ($purchaseStatus === 'partial') {
                    $itemStatus = 3; // Partially Paid
                }

                $productDetail = ProductDetailsModel::create([
                    'product_id' => $productData['product_id'],
                    'product_details_code' => $details_code,
                    'branch_id' => $branchId,
                    'name' => $product->name,
                    'native_name' => $product->native_name,
                    'code' => $product->code,
                    'category_id' => $product->category_id,
                    'supplier_id' => $productData['supplier_id'],
                    'unit_price' => $productData['unit_price'],
                    'margin_price' => $productData['margin_price'],
                    'qty' => $productData['qty'],
                    'status' => $itemStatus,
                    'total_amount' => $itemTotalAmount,
                    'description' => $productData['description'] ?? null,
                    'purchase_id' => $purchase->id, // 🔥 Link to purchase
                    'active_flag' => 'A'
                ]);

                // Update temp quantity
                $newTempQty = $tempQuantity->temp_qty - $productData['qty'];
                if ($newTempQty <= 0) {
                    $tempQuantity->delete();
                } else {
                    $tempQuantity->update(['temp_qty' => $newTempQty]);
                }

                $processedProducts[] = [
                    'product_id' => $productData['product_id'],
                    'product_name' => $product->name,
                    'supplier_id' => $productData['supplier_id'],
                    'qty_purchased' => $productData['qty'],
                    'total_amount' => $itemTotalAmount,
                    'payment_status' => $itemStatus
                ];

                Log::info("Processed product: {$product->name} (ID: {$product->id}) - Qty: {$productData['qty']}");
            }

            // 🔥 CREATE PURCHASE PAYMENT RECORD (if payment amount provided)
            if ($paidAmount > 0 && $request->payment_type) {
                ProductPaymentModel::create([
                    'product_dtl_id' => null, // Not linked to specific product detail
                    'payment_type' => $request->payment_type,
                    'payment_date' => $request->payment_date ?? now()->toDateString(),
                    'total_amount' => $totalAmount,
                    'paid_amount' => $paidAmount,
                    'active_flag' => 'A',
                    'type' => 'purchase_payment',
                    'purchase_id' => $purchase->id // 🔥 Link payment to purchase
                ]);

                Log::info("Created purchase payment: ₹{$paidAmount} for purchase {$purchase->purchase_code}");
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Temporary quantities successfully converted to purchase',
                'purchase_data' => [
                    'purchase_id' => $purchase->id,
                    'purchase_code' => $purchase->purchase_code,
                    'purchase_date' => $purchase->purchase_date,
                    'total_amount' => $totalAmount,
                    'paid_amount' => $paidAmount,
                    'due_amount' => $dueAmount,
                    'status' => $purchaseStatus,
                ],
                'processed_products' => $processedProducts,
                'total_products_processed' => count($processedProducts),
                'payment_created' => $paidAmount > 0 ? true : false,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error converting temp to purchase: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to convert temporary quantities to purchase',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // 🔥 ADD THIS NEW HELPER METHOD
    private function generatePurchaseCode(): string
    {
        $lastPurchase = PurchaseModel::orderBy('id', 'desc')->first();

        if ($lastPurchase && $lastPurchase->purchase_code) {
            $lastCode = $lastPurchase->purchase_code;
            $parts = explode('-', $lastCode);
            $prefix = $parts[0];
            $number = isset($parts[1]) ? (int) $parts[1] : 0;
            $newNumber = str_pad($number + 1, 3, '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }

        return 'PUR-001';
    }


    private function generateProductDetailCode(): string
    {
        $lastDetail = ProductDetailsModel::orderBy('product_details_sid', 'desc')->first();

        if ($lastDetail && $lastDetail->product_details_code) {
            $lastCode = $lastDetail->product_details_code;
            $parts = explode('-', $lastCode);
            $prefix = $parts[0];
            $number = isset($parts[1]) ? (int) $parts[1] : 0;
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }

        return 'PROD-001';
    }

    public function show(Request $request, $id)
    {
        try {
            $user = Auth::user();

            // Build the query with relationships
            $query = OrdersModel::with([
                'client' => function ($query) {
                    $query->select('id', 'user_id');
                },
                'client.user' => function ($query) {
                    $query->select('id', 'name', 'phone');
                },
                'branch',
                'ordersItem' => function ($query) {
                    $query->select('id', 'order_id', 'product_id', 'product_name', 'quantity', 'available_quantity', 'price', 'total');
                }
            ])->where('id', $id);

            // Add branch filtering if provided
            if ($request->filled('branch_id')) {
                $query->where('branch_id', $request->branch_id);
            }

            // Client access restriction
            if ($user->user_flg === 'C') {
                $client_id = ClientModel::where('user_id', $user->id)
                    ->when($request->filled('branch_id'), function ($q) use ($request) {
                        $q->where('branch_id', $request->branch_id);
                    })
                    ->pluck('id')
                    ->first();

                if ($client_id) {
                    $query->where('client_id', $client_id);
                } else {
                    return response()->json([
                        'status' => false,
                        'message' => 'Invalid client/user',
                    ], 400);
                }
            }

            $order = $query->first();

            if (!$order) {
                return response()->json([
                    'status' => false,
                    'message' => 'Order not found',
                ], 404);
            }

            return response()->json([
                'status' => true,
                'order' => $order,
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching order details: ' . $e->getMessage());
            return response()->json([
                'status' => false,
                'message' => 'Failed to fetch order details',
                'error' => $e->getMessage()
            ], 500);
        }
    }
    public function checkClientByPhone(Request $request)
    {
        $request->validate([
            'phone' => 'required|string'
        ]);

        try {
            $user = User::where('phone', $request->phone)->first();

            if (!$user) {
                return response()->json([
                    'success' => false,
                    'user_exists' => false,
                    'message' => 'Phone number not found',
                    'client_name' => '',
                    'addresses' => []
                ]);
            }

            $client = ClientModel::where('user_id', $user->id)->first();

            $addresses = [];
            if ($client) {
                $addresses = ClientAddressModel::where('client_id', $client->id)
                    ->where('oprtnl_flag', 'A')
                    ->orderBy('id', 'desc')
                    ->get();
            }

            return response()->json([
                'success' => true,
                'user_exists' => true,
                'client_name' => $user->name,
                'client_id' => $client ? $client->id : null,
                'addresses' => $addresses
            ]);

        } catch (\Exception $e) {
            \Log::error('Error checking client by phone: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error checking client information'
            ], 500);
        }
    }

}
